/*
 * ImageTools.java
 *
 * Created on 9 april 2007, 18:07
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package iliadmanifestcreator;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Paint;
import java.awt.RenderingHints;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageFilter;
import java.awt.image.BufferedImageOp;
import java.awt.image.ByteLookupTable;
import java.awt.image.ColorConvertOp;
import java.awt.image.FilteredImageSource;
import java.awt.image.IndexColorModel;
import java.awt.image.LookupOp;
import java.awt.image.RescaleOp;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import javax.imageio.ImageIO;

/**
 *
 * @author Bert
 */
public class ImageTools {
    static byte[] baILiadColors = { (byte)0x00, (byte)0x11, (byte)0x22, (byte)0x33, (byte)0x44, (byte)0x55, (byte)0x66, (byte)0x77, (byte)0x88, (byte)0x88, (byte)0x99, (byte)0xAA, (byte)0xBB, (byte)0xCC, (byte)0xDD, (byte)0xEE, (byte)0xFF };
    
    /** Creates a new instance of ImageTools */
    public ImageTools() {
    }
    
    /**
     *Fits the provided image to the given size, keeping the ratio intact.
     */
    public static BufferedImage scaleImage(BufferedImage imgSource, int nFitWidth, int nFitHeight) {
        if (nFitWidth<=0 || nFitHeight<=0)
            throw new IllegalArgumentException("width and height must be larger than 0!");
        
        //calculate the new size
        int nImgWidth = imgSource.getWidth();
        int nImgHeight = imgSource.getHeight();
        double nImgRatio = 1.*nImgHeight/nImgWidth;
        double nFitRatio = 1.*nFitHeight/nFitWidth;
        
        int nHeight, nWidth;
        if (nImgRatio>nFitRatio) { //fit height
            nHeight = nFitHeight;
            nWidth = (int)(nFitHeight/nImgRatio);
        } else { //fit width
            nHeight = (int)(nFitWidth*nImgRatio);
            nWidth = nFitWidth;
        }
        
        //center image
        int nXPos = (nFitWidth - nWidth) / 2;
        int nYPos = (nFitHeight - nHeight) / 2;        
        
        //scale
        Image imgScaled = imgSource.getScaledInstance(nWidth, nHeight, java.awt.Image.SCALE_SMOOTH);
        BufferedImage biScaled = new BufferedImage(nFitWidth, nFitHeight, BufferedImage.TYPE_INT_ARGB);
        Graphics2D big = biScaled.createGraphics();
        big.drawImage(imgScaled, nXPos, nYPos, nWidth, nHeight, null);
        big.dispose();
        
        return biScaled;
    }
    
    /**
     *Replaces the alpha channel by the provided color
     */
    public static BufferedImage replaceAlphaByColor(BufferedImage imgSource, Color c) {
        int nWidth = imgSource.getWidth();
        int nHeight = imgSource.getHeight();
        
        BufferedImage biNew = new BufferedImage(nWidth, nHeight, BufferedImage.TYPE_INT_RGB);
        Graphics2D g = biNew.createGraphics();
        g.setColor(c);
        g.fillRect(0, 0, nWidth, nHeight);
        g.drawImage(imgSource, 0, 0, nWidth, nHeight, null);
        g.dispose();
        
        return biNew;
    }
    
    /**
     *Returns the inverted image of the source image.
     */
    public static BufferedImage negative(BufferedImage imgSource) {
        //note: we should make this static!
        byte reverse[] = new byte[256];
        for (int i = 0; i < 256; i++)
            reverse[i] = (byte) (255 - i);
        ByteLookupTable lookupTable = new ByteLookupTable(0, reverse);
        
        LookupOp lop = new LookupOp(lookupTable, null);
        return lop.filter(imgSource, null);
        
// doesn't work for some reason
//        RescaleOp rop = new RescaleOp(-1.f, 255.f, null);
//        return rop.filter(imgSource, null);
    }
    
    /**
     *Converts a BufferedImage to another BufferedImage using only the ILiad supported colors 
     * (0x000000, 0x111111, 0x222222, ..., 0xFFFFFF)
     */
    public static BufferedImage convertTo4bitGrayScale(BufferedImage imgSource) {
        //first convert to 'normal' grayscale - gives better result when reverting to 4-bit
        BufferedImage imgGray = convertToGrayScale(imgSource);
        
        //now convert to 4-bit gray-level
        IndexColorModel icm = new IndexColorModel(4, 16, baILiadColors, baILiadColors, baILiadColors);
        BufferedImage bi4Bit = new BufferedImage(imgSource.getWidth(), imgSource.getHeight(), BufferedImage.TYPE_BYTE_BINARY, icm);
        Graphics2D g = bi4Bit.createGraphics();
        g.drawRenderedImage(imgGray, null);
        g.dispose();
        
        return bi4Bit;
    }
    
    /**
     *Converts a BufferedImage to its grayscale equivalent.
     *It uses a correction algorithm because humans are more sensitive to certain colors 
     * the to others. (Check http://www.faqs.org/faqs/graphics/colorspace-faq/ for details)
     */
    public static BufferedImage convertToGrayScale(BufferedImage imgSource) {
        int nWidth = imgSource.getWidth();
        int nHeight = imgSource.getHeight();
        BufferedImage imgGray = new BufferedImage(nWidth, nHeight, BufferedImage.TYPE_BYTE_GRAY);
        java.awt.image.WritableRaster raster = imgGray.getRaster().createCompatibleWritableRaster();
        
        //Brightness = 0.212671 * R + 0.715160 * G + 0.072169 * B
        int rgb, r, g, b, gray;
        for (int y=0; y < nHeight; y++) {
            for (int x=0; x < nWidth; x++) {
                rgb = imgSource.getRGB(x, y);
                r = ((rgb & 0xff0000) >> 16);
                g = ((rgb & 0x00ff00) >> 8);
                b = (rgb & 0x0000ff);
                
                gray = (int)(0.212671 * r + 0.715160 * g + 0.072169 * b);
            
                raster.setSample(x, y, 0, gray);            
            }
        }
        
        imgGray.setData(raster);
        
        return imgGray;
    }
    
    //No longer used because even gray values are changed (why !?)
    public static BufferedImage convertToGrayScale_old(BufferedImage imgSource) {
        BufferedImageOp op = new ColorConvertOp(ColorSpace.getInstance(ColorSpace.CS_GRAY), null); 
        BufferedImage biTemp = op.filter(imgSource, null);
        
        return biTemp;
    }

    /**
     *Reads a BufferedImage from a File.<br>
     *Because ImageIO.read works painfully slow, apparently because ImageIO uses some 
     *strange custom image types. That's why we're forced to create a new BufferedImage
     *with an image type chosen by us.
     *<br>Example: loading a jpg image with ImageIO.read(): 21906 ms (on first use - not when calling 
     * read()), loading the same image with readImage(): 843 ms.
     *@param file File to read the image from
     *@result a BufferedImage object or <code>null</code> if an error occured.
     */
    public static BufferedImage readImage(File file){
        try {
            //read an image
            BufferedImage biOrg = ImageIO.read(file);
            int w = biOrg.getWidth();
            int h = biOrg.getHeight();
 
            //convert to a known format
            BufferedImage biNew = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = biNew.createGraphics();
            g.drawRenderedImage(biOrg, null);
            g.dispose();
            
            return biNew;
        } catch (IOException ex) {
            ex.printStackTrace();
        }
        
        return null;
    }

    /**
     *Writes the image in the given format to the file <code>strFile</code>
     *@param img Source BufferedImage
     *@param strFormat Format in which to save the image. eg: 'png', 'gif', 'bmp', 'jpg'
     *@param strFile Full path to the destination file
     *@result <code>true</code> if the file was saved successfully, <code>false</code> otherwise
     */
    public static boolean writeImage(BufferedImage img, String strFormat, String strFile) {
        try {
            return ImageIO.write(img, strFormat, new FileOutputStream(strFile));
        } catch (FileNotFoundException ex) {
            ex.printStackTrace();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
        
        return false;
    }
    
}
